/**
 * \file           DnsAutoDiscovery.h
 * \brief          Header for CDnsAutoDiscovery Class used for DNS and AutoDsicovery
 * \author         RJen
 * \date           11/29/2011
 **/

#ifndef _DNS_AUTO_DISCOVERY_H_
#define _DNS_AUTO_DISCOVERY_H_

// Determines which host name resolution mode to use first
enum HOST_NAME_RESOLUTION_MODE {
  HOST_NAME_RESOLUTION_MODE_UNKNOWN = 0,
  HOST_NAME_RESOLUTION_MODE_DNS = 1,
  HOST_NAME_RESOLUTION_MODE_AUTODISCOVERY = 2,
};

enum DNS_HOSTNAME_ERRORS {
  BAD_PTR = 0,
  INVALID_HOST_NAME,
  INVALID_DOMAIN_NAME,
  MEM_MANAGER_DOWN,
  UNABLE_TO_ALLOCATE_MEMORY,
};

class CDnsAutoDiscovery
{
  public:
    // Constructors and destructors
    CDnsAutoDiscovery(void); 
    ~CDnsAutoDiscovery(void); 
  
    // Initializes the driver and determines resolution mode
    bool Initialize(void); 
    
    // Gets IP address of a host using DNS and/or autodiscovery
    bool GetHostByName(const char *pHostName, UINT16 HostNameSize, UINT8 *pIpAddr, UINT8 IpAddrSize); 
    
    // Gets IP address of a host using DNS only
    bool GetHostByNameViaDns(const char *pHostName, UINT8 HostNameLen, UINT8 *pIpAddr);
    
    // Sets the domain name used to determine FQDN from regular hostnames
    void SetDomainName(const char *pDomainName, UINT16 DomainNameSize);
    
    // Should be called when the ethernet link does down
    void SignalLinkLoss(void);
    
    // Flush DNS cache
    void FlushDnsCache(void);
        
    // Checks to make sure hostname is valid (can handle FQDN), can pass in domain names too
    static UINT8 IsHostNameValid(const char *pHostName, UINT16 HostNameSize);
    
    HOST_NAME_RESOLUTION_MODE GetNameResolutionMode(void); // Gets the resolution mode
    void SetNameResolutionMode(HOST_NAME_RESOLUTION_MODE Mode); // Sets the resolution mode
    
    bool Initialized(void); // Is the object initialized

  private:
    HOST_NAME_RESOLUTION_MODE m_CurrentResolutionMode; // Current name resolution mode
    
    // Determines which mode to use first, DNS or Auto Discovery
    void DetermineNameResolutionMode(void);
    
    // Task for initializing the class object
    static void InitTaskWrapper(UINT32 Param);
    void InitTask(void);
    
    UINT32 m_hAccessLock;
    UINT32 m_hInitTaskHandle;
    UINT8* m_pDomainName;
    UINT8  m_DomainNameLen;
    bool   m_bInvalidEthParams;
};

extern CDnsAutoDiscovery* g_pDnsAutoDiscovery;

// DNS related command functions
extern INT32 ListDnsCmd(UINT32 argc, char* cmd);
extern INT32 AddDnsCmd(UINT32 argc, char* cmd);
extern INT32 RemDnsCmd(UINT32 argc, char* cmd);
extern INT32 TestDnsCmd(UINT32 argc, char* cmd);
extern INT32 DomainNameCmd(UINT32 argc, char* cmd);
extern INT32 DhcpOptionsCmd(UINT32 argc, char* cmd);
extern INT32 ResolutionModeCmd(UINT32 argc, char* cmd);
extern INT32 ResolveHostnameCmd(UINT32 argc, char* cmd);
#endif // #ifndef _DNS_AUTO_DISCOVERY_H_